#include <p.h>

/* Path Integral Monte Carlo code developed by A. Balaz 
   (antun@ipb.ac.rs) for papers:
         
   "Systematic Speedup in the Convergence of Path Integrals"
   by A. Bogojevic, A. Balaz, and A. Belic
   Phys. Rev. Lett. 94 (2005) 180403

   "Fast Converging Path Integrals for Time-Dependent Potentials"
   by A. Balaz, I. Vidanovic, A. Bogojevic, A, Pelster
   arXiv:0912.2743

   Address: Institute of Physics, Belgrade, Serbia
   Scientific Computing Laboratory, http://www.scl.rs/speedup/

   Public use and modification of this code is allowed providing 
   the above paper is properly acknowledged. The author would be 
   grateful for all information and/or comments regarding the use   
   of the code.  
*/

/* USER CONFIGURABLE SECTION:

   1) Defining the level parameter p (1 <= p <= 8). 
   2) Supplying the C function for the evaluation of the potential V
      and its parameters.
*/

/* 1) Defining the level parameter p (1 <= p <= 8)

      The level parameter p corresponds to the function Vp()
      (improved potential) which will be compiled. You need to adjust
      the following preprocessor variable. The value below is for p = 4.
*/

#define _p 4

/* 2) Supplying the C function for the evaluation of the potential V
      and its parameters.

      Function V0(x) provides the value of the potential parameters,
      used later to calculate the effective potential.

      Parameters of the potential are par[0] ..., as entered by
      you on the command line. $\phi^4$ example is given below.
*/

void V0(double x, double t) /* Example of the user supplied function V0() for the harmonic oscillator rescaled with the Grosche factor sqrt(t*t+1). */
{
   xs[1] = x;
   xs[2] = x * x;
   ts[1] = t;
   ftinv[1] = 1 / (t * t + 1);
   ftinv[2] = ftinv[1] * ftinv[1];

   return;
}

/* END OF THE USER CONFIGURABLE SECTION! */


/* Function func(q[]) returns exp(- action + norm), where the variable action is
   the action for the given path (q[0], q[step], ..., q[N]) plus the exponent of
   the Gaussian probability density function (*distrexp), and norm is the normalization
   (*norm) of the path integral (which also includes the normalization of Gaussians).
   Array *norm is defined in main(), array *distrexp in distr().
   The improved potential Vp() for every pair of points (q[i], q[i + step])
   is evaluated at the mid-point qbar.
*/

double func(double *q, int s, int n)
{
   void V0(double, double); /* The user supplied function that returns the unimproved potential
                               and its derivatives. */
   double Vp(void); /* The function that, for a given p, returns the improved potential. */
   double tbar; /* Time mid-point. */

   int i; /* Loop variable for the loop over the time steps. */
   int j; /* Loop variable for the loop defining deltas array. */

   action = 0; /* Initialiation of the action. */
   Nmax = two[s]; /* The maximal number of time steps. */
   Nc = two[n]; /* The current number of time steps. */
   step = two[s - n]; /* Index distance of path nodes at current bisection level. */
   epss[1] = T / Nc; /* Current time step and related variables. */
   epsinv = 1. / epss[1];
   tbar = ta - 0.5 * epss[1];
   for(i = 2; i < _p; i++) epss[i] = epss[1] * epss[i - 1];

   for(i = 0; i < Nmax; i += step) /* Loop over path nodes at current bisection level. */
   {
      tbar += epss[1];
      qbar = (q[i + step] + q[i]) * 0.5; /* Mid-point for nearest nodes. */
      deltas[1] = 0.5 * (q[i + step] - q[i]); /* Half-distance between nearest nodes. */
      deltas[2] = deltas[1] * deltas[1];
      for(j = 3; j < 2 * _p - 1; j++) deltas[j] = deltas[1] * deltas[j - 1];
      V0(qbar, tbar); /* Initialization of unimproved potential and its derivatives at qbar. */
      action += 2 * deltas[2] * epsinv + epss[1] * Vp() - distrexp[i]; /* Accumulation of the action minus
                                                                     the exponent of the Gaussian. */
   }

   return exp(- action + norm[n]); /* norm[n] includes the path integral norm for a
                                      given number of time steps 2^n and normalization of Gaussians
                                      used as probability density functions. */
}

/* For a chosen p, the function Vp() returns the value of the improved potential. */

#if _1p == 1
double Vp(void)
{
   return (xs[2]*ftinv[2])/2;
}
#endif

#if _p == 2
double Vp(void)
{
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12;           
}
#endif

#if _p == 3
double Vp(void)
{
   ts[2] = ts[1] * ts[1];
   ftinv[3] = ftinv[2] * ftinv[1];
   ftinv[4] = ftinv[3] * ftinv[1];
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12 - (xs[2]*epss[2]*ftinv[3])/12 - (xs[2]*epss[2]*ftinv[4])/24 - 
 (2*xs[1]*deltas[1]*epss[1]*ftinv[3]*ts[1])/3 + (xs[2]*epss[2]*ftinv[4]*ts[2])/2;           
}
#endif

#if _p == 4
double Vp(void)
{
   ts[2] = ts[1] * ts[1];
   ftinv[3] = ftinv[2] * ftinv[1];
   ftinv[4] = ftinv[3] * ftinv[1];
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12 - (xs[2]*epss[2]*ftinv[3])/12 - (deltas[2]*epss[2]*ftinv[3])/20 - (epss[3]*ftinv[3])/120 - 
 (xs[2]*epss[2]*ftinv[4])/24 - (deltas[2]*epss[2]*ftinv[4])/360 - (epss[3]*ftinv[4])/360 - (2*xs[1]*deltas[1]*epss[1]*ftinv[3]*ts[1])/3 + 
 (xs[2]*epss[2]*ftinv[4]*ts[2])/2 + (3*deltas[2]*epss[2]*ftinv[4]*ts[2])/10 + (epss[3]*ftinv[4]*ts[2])/20;           
}
#endif

#if _p == 5
double Vp(void)
{
   ts[2] = ts[1] * ts[1];
   ts[3] = ts[2] * ts[1];
   ts[4] = ts[3] * ts[1];
   ftinv[3] = ftinv[2] * ftinv[1];
   ftinv[4] = ftinv[3] * ftinv[1];
   ftinv[5] = ftinv[4] * ftinv[1];
   ftinv[6] = ftinv[5] * ftinv[1];
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12 - (xs[2]*epss[2]*ftinv[3])/12 - (deltas[2]*epss[2]*ftinv[3])/20 - (epss[3]*ftinv[3])/120 - 
 (xs[2]*epss[2]*ftinv[4])/24 - (deltas[2]*epss[2]*ftinv[4])/360 - (epss[3]*ftinv[4])/360 + (3*xs[2]*epss[4]*ftinv[4])/160 + (xs[2]*epss[4]*ftinv[5])/120 + 
 (xs[2]*epss[4]*ftinv[6])/240 - (2*xs[1]*deltas[1]*epss[1]*ftinv[3]*ts[1])/3 + (3*xs[1]*deltas[1]*epss[3]*ftinv[4]*ts[1])/10 + 
 (xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[1])/45 + (xs[2]*epss[2]*ftinv[4]*ts[2])/2 + (3*deltas[2]*epss[2]*ftinv[4]*ts[2])/10 + (epss[3]*ftinv[4]*ts[2])/20 - 
 (3*xs[2]*epss[4]*ftinv[5]*ts[2])/10 - (11*xs[2]*epss[4]*ftinv[6]*ts[2])/180 - (4*xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[3])/5 + (xs[2]*epss[4]*ftinv[6]*ts[4])/2;           
}
#endif

#if _p == 6
double Vp(void)
{
   ts[2] = ts[1] * ts[1];
   ts[3] = ts[2] * ts[1];
   ts[4] = ts[3] * ts[1];
   ftinv[3] = ftinv[2] * ftinv[1];
   ftinv[4] = ftinv[3] * ftinv[1];
   ftinv[5] = ftinv[4] * ftinv[1];
   ftinv[6] = ftinv[5] * ftinv[1];
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12 - (xs[2]*epss[2]*ftinv[3])/12 - (deltas[2]*epss[2]*ftinv[3])/20 - (epss[3]*ftinv[3])/120 - 
 (xs[2]*epss[2]*ftinv[4])/24 - (deltas[2]*epss[2]*ftinv[4])/360 - (epss[3]*ftinv[4])/360 + (3*xs[2]*epss[4]*ftinv[4])/160 + (3*deltas[2]*epss[4]*ftinv[4])/224 + 
 (3*epss[5]*ftinv[4])/2240 + (xs[2]*epss[4]*ftinv[5])/120 + (deltas[2]*epss[4]*ftinv[5])/840 + (epss[5]*ftinv[5])/2520 + (xs[2]*epss[4]*ftinv[6])/240 + 
 (deltas[2]*epss[4]*ftinv[6])/15120 + (epss[5]*ftinv[6])/5670 - (2*xs[1]*deltas[1]*epss[1]*ftinv[3]*ts[1])/3 + (3*xs[1]*deltas[1]*epss[3]*ftinv[4]*ts[1])/10 + 
 (xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[1])/45 + (xs[1]*deltas[3]*epss[3]*ftinv[5]*ts[1])/252 + (xs[1]*deltas[1]*epss[4]*ftinv[5]*ts[1])/2016 + 
 (xs[2]*epss[2]*ftinv[4]*ts[2])/2 + (3*deltas[2]*epss[2]*ftinv[4]*ts[2])/10 + (epss[3]*ftinv[4]*ts[2])/20 - (3*xs[2]*epss[4]*ftinv[5]*ts[2])/10 - 
 (3*deltas[2]*epss[4]*ftinv[5]*ts[2])/14 - (3*epss[5]*ftinv[5]*ts[2])/140 - (11*xs[2]*epss[4]*ftinv[6]*ts[2])/180 - 
 (19*deltas[2]*epss[4]*ftinv[6]*ts[2])/1260 - (epss[5]*ftinv[6]*ts[2])/315 - (4*xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[3])/5 + (xs[2]*epss[4]*ftinv[6]*ts[4])/2 + 
 (5*deltas[2]*epss[4]*ftinv[6]*ts[4])/14 + (epss[5]*ftinv[6]*ts[4])/28;           
}
#endif

#if _p == 7
double Vp(void)
{
   ts[2] = ts[1] * ts[1];
   ts[3] = ts[2] * ts[1];
   ts[4] = ts[3] * ts[1];
   ts[5] = ts[4] * ts[1];
   ts[6] = ts[5] * ts[1];
   ftinv[3] = ftinv[2] * ftinv[1];
   ftinv[4] = ftinv[3] * ftinv[1];
   ftinv[5] = ftinv[4] * ftinv[1];
   ftinv[6] = ftinv[5] * ftinv[1];
   ftinv[7] = ftinv[6] * ftinv[1];
   ftinv[8] = ftinv[7] * ftinv[1];
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12 - (xs[2]*epss[2]*ftinv[3])/12 - (deltas[2]*epss[2]*ftinv[3])/20 - (epss[3]*ftinv[3])/120 - 
 (xs[2]*epss[2]*ftinv[4])/24 - (deltas[2]*epss[2]*ftinv[4])/360 - (epss[3]*ftinv[4])/360 + (3*xs[2]*epss[4]*ftinv[4])/160 + (3*deltas[2]*epss[4]*ftinv[4])/224 + 
 (3*epss[5]*ftinv[4])/2240 + (xs[2]*epss[4]*ftinv[5])/120 + (deltas[2]*epss[4]*ftinv[5])/840 + (epss[5]*ftinv[5])/2520 - (xs[2]*epss[6]*ftinv[5])/224 + 
 (xs[2]*epss[4]*ftinv[6])/240 + (deltas[2]*epss[4]*ftinv[6])/15120 + (epss[5]*ftinv[6])/5670 - (37*xs[2]*epss[6]*ftinv[6])/20160 - 
 (11*xs[2]*epss[6]*ftinv[7])/10080 - (17*xs[2]*epss[6]*ftinv[8])/40320 - (2*xs[1]*deltas[1]*epss[1]*ftinv[3]*ts[1])/3 + (3*xs[1]*deltas[1]*epss[3]*ftinv[4]*ts[1])/10 + 
 (xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[1])/45 + (xs[1]*deltas[3]*epss[3]*ftinv[5]*ts[1])/252 + (xs[1]*deltas[1]*epss[4]*ftinv[5]*ts[1])/2016 - 
 (3*xs[1]*deltas[1]*epss[5]*ftinv[5]*ts[1])/28 - (13*xs[1]*deltas[1]*epss[5]*ftinv[6]*ts[1])/1260 - (xs[1]*deltas[1]*epss[5]*ftinv[7]*ts[1])/630 + 
 (xs[2]*epss[2]*ftinv[4]*ts[2])/2 + (3*deltas[2]*epss[2]*ftinv[4]*ts[2])/10 + (epss[3]*ftinv[4]*ts[2])/20 - (3*xs[2]*epss[4]*ftinv[5]*ts[2])/10 - 
 (3*deltas[2]*epss[4]*ftinv[5]*ts[2])/14 - (3*epss[5]*ftinv[5]*ts[2])/140 - (11*xs[2]*epss[4]*ftinv[6]*ts[2])/180 - 
 (19*deltas[2]*epss[4]*ftinv[6]*ts[2])/1260 - (epss[5]*ftinv[6]*ts[2])/315 + (15*xs[2]*epss[6]*ftinv[6]*ts[2])/112 + (29*xs[2]*epss[6]*ftinv[7]*ts[2])/840 + 
 (17*xs[2]*epss[6]*ftinv[8]*ts[2])/2160 - (4*xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[3])/5 + (5*xs[1]*deltas[1]*epss[5]*ftinv[6]*ts[3])/7 + 
 (4*xs[1]*deltas[1]*epss[5]*ftinv[7]*ts[3])/105 + (xs[2]*epss[4]*ftinv[6]*ts[4])/2 + (5*deltas[2]*epss[4]*ftinv[6]*ts[4])/14 + (epss[5]*ftinv[6]*ts[4])/28 - 
 (15*xs[2]*epss[6]*ftinv[7]*ts[4])/28 - (61*xs[2]*epss[6]*ftinv[8]*ts[4])/840 - (6*xs[1]*deltas[1]*epss[5]*ftinv[7]*ts[5])/7 + (xs[2]*epss[6]*ftinv[8]*ts[6])/2;           
}
#endif

#if _p == 8
double Vp(void)
{
   ts[2] = ts[1] * ts[1];
   ts[3] = ts[2] * ts[1];
   ts[4] = ts[3] * ts[1];
   ts[5] = ts[4] * ts[1];
   ts[6] = ts[5] * ts[1];
   ftinv[3] = ftinv[2] * ftinv[1];
   ftinv[4] = ftinv[3] * ftinv[1];
   ftinv[5] = ftinv[4] * ftinv[1];
   ftinv[6] = ftinv[5] * ftinv[1];
   ftinv[7] = ftinv[6] * ftinv[1];
   ftinv[8] = ftinv[7] * ftinv[1];
   return (xs[2]*ftinv[2])/2 + (deltas[2]*ftinv[2])/6 + (epss[1]*ftinv[2])/12 - (xs[2]*epss[2]*ftinv[3])/12 - (deltas[2]*epss[2]*ftinv[3])/20 - (epss[3]*ftinv[3])/120 - 
 (xs[2]*epss[2]*ftinv[4])/24 - (deltas[2]*epss[2]*ftinv[4])/360 - (epss[3]*ftinv[4])/360 + (3*xs[2]*epss[4]*ftinv[4])/160 + (3*deltas[2]*epss[4]*ftinv[4])/224 + 
 (3*epss[5]*ftinv[4])/2240 + (xs[2]*epss[4]*ftinv[5])/120 + (deltas[2]*epss[4]*ftinv[5])/840 + (epss[5]*ftinv[5])/2520 - (xs[2]*epss[6]*ftinv[5])/224 - 
 (deltas[2]*epss[6]*ftinv[5])/288 - (epss[7]*ftinv[5])/4032 + (xs[2]*epss[4]*ftinv[6])/240 + (deltas[2]*epss[4]*ftinv[6])/15120 + (epss[5]*ftinv[6])/5670 - 
 (37*xs[2]*epss[6]*ftinv[6])/20160 - (13*deltas[2]*epss[6]*ftinv[6])/33600 - (epss[7]*ftinv[6])/14400 - (11*xs[2]*epss[6]*ftinv[7])/10080 - 
 (17*deltas[2]*epss[6]*ftinv[7])/453600 - (epss[7]*ftinv[7])/28350 - (17*xs[2]*epss[6]*ftinv[8])/40320 - (deltas[2]*epss[6]*ftinv[8])/604800 - 
 (epss[7]*ftinv[8])/75600 - (2*xs[1]*deltas[1]*epss[1]*ftinv[3]*ts[1])/3 + (3*xs[1]*deltas[1]*epss[3]*ftinv[4]*ts[1])/10 + 
 (xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[1])/45 + (xs[1]*deltas[3]*epss[3]*ftinv[5]*ts[1])/252 + (xs[1]*deltas[1]*epss[4]*ftinv[5]*ts[1])/2016 - 
 (3*xs[1]*deltas[1]*epss[5]*ftinv[5]*ts[1])/28 - (13*xs[1]*deltas[1]*epss[5]*ftinv[6]*ts[1])/1260 - (xs[1]*deltas[3]*epss[5]*ftinv[6]*ts[1])/432 - 
 (xs[1]*deltas[1]*epss[6]*ftinv[6]*ts[1])/4608 - (xs[1]*deltas[1]*epss[5]*ftinv[7]*ts[1])/630 - (xs[1]*deltas[3]*epss[5]*ftinv[7]*ts[1])/3780 - 
 (17*xs[1]*deltas[1]*epss[6]*ftinv[7]*ts[1])/483840 + (xs[2]*epss[2]*ftinv[4]*ts[2])/2 + (3*deltas[2]*epss[2]*ftinv[4]*ts[2])/10 + (epss[3]*ftinv[4]*ts[2])/20 - 
 (3*xs[2]*epss[4]*ftinv[5]*ts[2])/10 - (3*deltas[2]*epss[4]*ftinv[5]*ts[2])/14 - (3*epss[5]*ftinv[5]*ts[2])/140 - (11*xs[2]*epss[4]*ftinv[6]*ts[2])/180 - 
 (19*deltas[2]*epss[4]*ftinv[6]*ts[2])/1260 - (epss[5]*ftinv[6]*ts[2])/315 + (15*xs[2]*epss[6]*ftinv[6]*ts[2])/112 + (5*deltas[2]*epss[6]*ftinv[6]*ts[2])/48 + 
 (5*epss[7]*ftinv[6]*ts[2])/672 + (29*xs[2]*epss[6]*ftinv[7]*ts[2])/840 + (47*deltas[2]*epss[6]*ftinv[7]*ts[2])/4200 + (epss[7]*ftinv[7]*ts[2])/700 + 
 (17*xs[2]*epss[6]*ftinv[8]*ts[2])/2160 + (533*deltas[2]*epss[6]*ftinv[8]*ts[2])/680400 + (xs[2]*deltas[2]*epss[6]*ftinv[8]*ts[2])/4536 + 
 (23*epss[7]*ftinv[8]*ts[2])/85050 + (5*xs[2]*epss[7]*ftinv[8]*ts[2])/290304 - (4*xs[1]*deltas[1]*epss[3]*ftinv[5]*ts[3])/5 + 
 (5*xs[1]*deltas[1]*epss[5]*ftinv[6]*ts[3])/7 + (4*xs[1]*deltas[1]*epss[5]*ftinv[7]*ts[3])/105 + (xs[1]*deltas[3]*epss[5]*ftinv[7]*ts[3])/108 + 
 (xs[1]*deltas[1]*epss[6]*ftinv[7]*ts[3])/1152 + (xs[2]*epss[4]*ftinv[6]*ts[4])/2 + (5*deltas[2]*epss[4]*ftinv[6]*ts[4])/14 + (epss[5]*ftinv[6]*ts[4])/28 - 
 (15*xs[2]*epss[6]*ftinv[7]*ts[4])/28 - (5*deltas[2]*epss[6]*ftinv[7]*ts[4])/12 - (5*epss[7]*ftinv[7]*ts[4])/168 - (61*xs[2]*epss[6]*ftinv[8]*ts[4])/840 - 
 (37*deltas[2]*epss[6]*ftinv[8]*ts[4])/1400 - (13*epss[7]*ftinv[8]*ts[4])/4200 - (6*xs[1]*deltas[1]*epss[5]*ftinv[7]*ts[5])/7 + (xs[2]*epss[6]*ftinv[8]*ts[6])/2 + 
 (7*deltas[2]*epss[6]*ftinv[8]*ts[6])/18 + (epss[7]*ftinv[8]*ts[6])/36;           
}
#endif
