/* NO USER CONFIGURABLE PARAMETERS HERE! */

/* Path Integral Monte Carlo code developed by A. Balaz 
   (antun@ipb.ac.rs) for papers:
         
   "Systematic Speedup in the Convergence of Path Integrals"
   by A. Bogojevic, A. Balaz, and A. Belic
   Phys. Rev. Lett. 94 (2005) 180403

   "Fast Converging Path Integrals for Time-Dependent Potentials"
   by A. Balaz, I. Vidanovic, A. Bogojevic, A, Pelster
   arXiv:0912.2743

   Address: Institute of Physics, Belgrade, Serbia
   Scientific Computing Laboratory, http://www.scl.rs/speedup/

   Public use and modification of this code is allowed providing 
   the above paper is properly acknowledged. The author would be 
   grateful for all information and/or comments regarding the use   
   of the code.  
*/

/* This file contains code from the Numerical Recipes that is used in our simulation. */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#define NR_END 1
#define FREE_ARG char*

int *ivector(long, long);
double *dvector(long, long);
double **dmatrix(long, long, long, long);
double ***d3tensor(long, long, long, long, long, long);
void free_ivector(int *, long, long);
void free_dvector(double *, long, long);
void free_dmatrix(double **, long, long, long, long);
void free_d3tensor(double ***, long, long, long, long, long, long);
void nrerror(char *);

int *ivector(long nl, long nh)
/* allocate an int vector with subscript range v[nl..nh] */
{
   int *v;

   v = (int *)malloc((size_t) ((nh - nl + 1 + NR_END) * sizeof(int)));
   if(!v) nrerror("allocation failure in ivector()");
   return v - nl + NR_END;
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

double *dvector(long nl, long nh)
/* allocate a double vector with subscript range v[nl..nh] */
{
   double *v;

   v = (double *)malloc((size_t) ((nh - nl + 1 + NR_END) * sizeof(double)));
   if(!v) nrerror("allocation failure in ldvector()");
   return v - nl + NR_END;
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

double **dmatrix(long nrl, long nrh, long ncl, long nch)
/* allocate a double matrix with subscript range m[nrl..nrh][ncl..nch] */
{
   long i, nrow = nrh - nrl + 1, ncol = nch - ncl + 1;
   double **m;

/* allocate pointers to rows */
   m = (double **) malloc((size_t)((nrow + NR_END) * sizeof(double*)));
   if(!m) nrerror("allocation failure 1 in dmatrix()");
   m += NR_END;
   m -= nrl;

/* allocate rows and set pointers to them */
   m[nrl] = (double *) malloc((size_t)((nrow * ncol + NR_END) * sizeof(double)));
   if(!m[nrl]) nrerror("allocation failure 2 in dmatrix()");
   m[nrl] += NR_END;
   m[nrl] -= ncl;

   for(i = nrl + 1; i <= nrh; i++) m[i] = m[i - 1] + ncol;

/* return pointer to array of pointers to rows */
   return m;
}

double ***d3tensor(long nrl, long nrh, long ncl, long nch, long ndl, long ndh)
/* allocate a double 3tensor with range t[nrl..nrh][ncl..nch][ndl..ndh] */
{
   long i, j, nrow = nrh - nrl + 1, ncol = nch - ncl + 1, ndep = ndh - ndl + 1;
   double ***t;

/* allocate pointers to pointers to rows */
   t = (double ***) malloc((size_t)((nrow + NR_END) * sizeof(double**)));
   if (!t) nrerror("allocation failure 1 in d3tensor()");
   t += NR_END;
   t -= nrl;

/* allocate pointers to rows and set pointers to them */
   t[nrl] = (double **) malloc((size_t)((nrow * ncol + NR_END) * sizeof(double*)));
   if (!t[nrl]) nrerror("allocation failure 2 in d3tensor()");
   t[nrl] += NR_END;
   t[nrl] -= ncl;

/* allocate rows and set pointers to them */
   t[nrl][ncl]=(double *) malloc((size_t)((nrow * ncol * ndep + NR_END) * sizeof(double)));
   if (!t[nrl][ncl]) nrerror("allocation failure 3 in d3tensor()");
   t[nrl][ncl] += NR_END;
   t[nrl][ncl] -= ndl;

   for(j = ncl + 1; j <= nch; j++) t[nrl][j] = t[nrl][j - 1] + ndep;

   for(i = nrl + 1; i <= nrh; i++)
   {
      t[i] = t[i - 1] + ncol;
      t[i][ncl] = t[i - 1][ncl] + ncol * ndep;
      for(j = ncl + 1; j <= nch; j++) t[i][j] = t[i][j - 1] + ndep;
   }

/* return pointer to array of pointers to rows */
   return t;
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

void free_ivector(int *v, long nl, long nh)
/* free an int vector allocated with ivector() */
{
   free((FREE_ARG) (v + nl - NR_END));
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

void free_dvector(double *v, long nl, long nh)
/* free a double vector allocated with dvector() */
{
   free((FREE_ARG) (v + nl - NR_END));
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

void free_dmatrix(double **m, long nrl, long nrh, long ncl, long nch)
/* free a double matrix allocated by dmatrix() */
{
   free((FREE_ARG) (m[nrl] + ncl - NR_END));
   free((FREE_ARG) (m + nrl - NR_END));
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

void free_d3tensor(double ***t, long nrl, long nrh, long ncl, long nch, long ndl, long ndh)
/* free a double d3tensor allocated by d3tensor() */
{
   free((FREE_ARG) (t[nrl][ncl] + ndl - NR_END));
   free((FREE_ARG) (t[nrl] + ncl - NR_END));
   free((FREE_ARG) (t + nrl - NR_END));
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */

void nrerror(char *error_text)
/* Numerical Recipes standard error handler */
{
   fprintf(stderr, "Numerical Recipes run-time error...\n");
   fprintf(stderr, "%s\n",error_text);
   fprintf(stderr, "...now exiting to system...\n");
   exit(EXIT_FAILURE);
}
/* (C) Copr. 1986-92 Numerical Recipes Software. */
