/* NO USER CONFIGURABLE PARAMETERS HERE! */

/* Path Integral Monte Carlo code developed by A. Balaz 
   (antun@ipb.ac.rs) for papers:
         
   "Systematic Speedup in the Convergence of Path Integrals"
   by A. Bogojevic, A. Balaz, and A. Belic
   Phys. Rev. Lett. 94 (2005) 180403

   "Fast Converging Path Integrals for Time-Dependent Potentials"
   by A. Balaz, I. Vidanovic, A. Bogojevic, A, Pelster
   arXiv:0912.2743

   Address: Institute of Physics, Belgrade, Serbia
   Scientific Computing Laboratory, http://www.scl.rs/speedup/

   Public use and modification of this code is allowed providing 
   the above paper is properly acknowledged. The author would be 
   grateful for all information and/or comments regarding the use   
   of the code.  
*/

/* The function main() reads command line parameters, initializes variables,
   calls the Monte Carlo function mc(), and prints the output.
*/

#include <main.h>

int main(int argc, char **argv)
{
   double a, b; /* Initial and final position */

   int s, n; /* Maximal number of time steps is 2^s; n is the loop variable that
                runs from 0 (or 1) to s, and corresponds to the current
                number of time steps 2^n. */

   double Nmc; /* Nmc is the number of Monte Carlo samples. */

   long seed; /* Initialization seed for the SPRNG random number generator. */

   int npar; /* Number of parameters of the potential enetered by the user. */

   double *amp, *err; /* Arrays of amplitudes and Monte Carlo errors for the
                         numbers of time steps 2^0, ..., 2^s. */

   double *q; /* Array that will hold the generated paths. */

   int i; /* Loop variable used during the initialization of *norm, ***Vd, *epss, *deltas. */
   int j, k; /* Further loop variables used during the initialization of ***Vd. */

   double tcpu; /* Overall time of execution. */

   struct timeval tv;  /* Structures defined in sys/time.h, used for the calculation of */
   struct timezone tz; /* the execution time. */

   if(argc < 8) /* Prints the usage if the number of */
   {            /* command line arguments is < 8. */
      fprintf(stderr, "Usage: %s ta a tb b s Nmc seed par[0] ...\n", argv[0]);
      fprintf(stderr, "This simulation calculates the probability amplitude\n");
      fprintf(stderr, "in imaginary time for the transition from the\n");
      fprintf(stderr, "initial state q(ta)=a to the final state q(tb)=b.\n");
      fprintf(stderr, "The number of time steps N will be 2^0, ..., 2^s.\n");
      fprintf(stderr, "Nmc is the number of MC samples\n");
      fprintf(stderr, "seed is used by the SPRNG (random number generator).\n");
      fprintf(stderr, "par[0], ... are additional parameters defining the potential\n");
      fprintf(stderr, "(initialize as many as needed and use them in the same order\n");
      fprintf(stderr, "in the user supplied function V0 in file p.c)\n");
      exit(EXIT_FAILURE);
   }

   ta = atof(argv[1]); /* Initialization of variables entered on the command line. */
   a = atof(argv[2]);
   tb = atof(argv[3]);
   b = atof(argv[4]);
   s = atol(argv[5]);
   Nmc = atof(argv[6]);

   seed = atol(argv[7]);

   par = dvector(0, argc - 8); /* Allocation of memory for the parameters of the potential. */

   for(npar = 0; npar < argc - 8; npar++) par[npar] = atof(argv[8 + npar]); /* Initialization of *par. */

   T = tb - ta; /* Time of propagation. */

   dpi = 8. * atan(1.); /* Initialization of dpi = 2 pi = 8 * atan(1). */

   amp = dvector(0, s); /* Allocation of memory for amplitudes. */

   err = dvector(0, s); /* Allocation of memory for errors. */

   two = ivector(0, s); /* Allocation of memory for array with powers of 2. */

   norm = dvector(0, s); /* Allocation of memory for norms of discretized path integrals
                            with the number of time steps 2^0, ..., 2^s, used in func(). */

   two[0] = 1; /* Initialization of array with powers of 2. */

   for(n = 1; n <= s; n ++)
   {
      two[n] = 2 * two[n - 1];
   }

   distrexp = dvector(0, two[s]); /* Allocation of memory for exponents of the Gaussian probability
                                     distribution functions, used in func(). */

   q = dvector(0, two[s]); /* Allocation of memory for paths. */

   dsigma2 = dvector(0, s); /* Allocation of memory for array of 2 sigma^2 values. Sigmas are
                               standard deviations of Gaussians used as probability density functions. */

   dsigma2inv = dvector(0, s); /* Allocation of memory for array of 1 / (2 sigma^2) values. */

   for(n = 0; n <= s; n++) /* Initialization of arrays *amp and *err to zeros. */
   {
      amp[n] = 0;
      err[n] = 0;
   }

   deltas = dvector(0, 2 * MAXDER); /* Array of powers of delta. */
   for(i = 0; i <= 2 * MAXDER; i++) deltas[i] = 0;

   epss = dvector(0, MAXDER); /* Array of powers of eps. */
   epsinv = 0;
   for(i = 0; i <= MAXDER; i++) epss[i] = 0;

   ftinv = dvector(0, MAXDER); /* Array of powers of inverse Grosche factor. */
   for(i = 0; i <= MAXDER; i++) ftinv[i] = 0;

   ts = dvector(0, MAXDER); /* Array of powers of t. */
   for(i = 0; i <= MAXDER; i++) ts[i] = 0;

   xs = dvector(0, MAXDER); /* Array of powers of x. */
   for(i = 0; i <= MAXDER; i++) xs[i] = 0;

   for(n = 0; n <= s; n++) /* Initialization of array of 2 sigma^2 and 1 / (2 sigma^2) values, */
   {                        /* where sigma^2 = eps = T / 2^n. */
      dsigma2[n] = T / two[n];
      dsigma2inv[n] = 1. / dsigma2[n];
   }

   for(n = 0; n <= s; n++) /* Initialization of array of the logarithms of norms. Norm is a product */
   {                        /* of (2 pi T / 2^n)^(- 2^n / 2) (normalization of the path integral in
                               the discretization with 2^n time steps) and normalization of Gaussians, i.e.
                               the square root of appropriate 2 pi sigma^2 for each path node. */
      norm[n] = - 0.5 * log(dpi) - 0.5 * two[n] * log(T / two[n]);

      for(i = 1; i <= n; i ++) norm[n] += 0.5 * two[i - 1] * log(0.5 * dsigma2[i]);
   }

   q[0] = a; /* Boundary conditions. */
   q[two[s]] = b;

   distrexp[0] = 0; /* q[0] is not generated, so distrexp[0] = 0. It is introduced for simplicity in distr(). */

   streamnum = 0; /* Initialization of the SPRNG random number generator. */
   nstreams = 1;
   stream = init_sprng(SPRNG_CMRG, streamnum, nstreams, seed, SPRNG_DEFAULT);

   /* Printing the header of the output. */
   fprintf(stdout, "N amplitude error time[s]\n");
   fflush(stdout);

   gettimeofday(&tv, &tz);               /* The initial time. Saving it here, we will avoid taking */
   tcpu = tv.tv_sec + tv.tv_usec * 1e-6; /* into account the time spent on the initialization. */

   mc(Nmc, q, amp, err, s); /* Calling the function that will carry out the Monte Carlo calculation. */

   gettimeofday(&tv, &tz);                      /* Calculates the time spent in mc(), as the */
   tcpu = tv.tv_sec + tv.tv_usec * 1e-6 - tcpu; /* difference between the current and initial time. */

   /* Processing the MC data. */

   amp[0] /= Nmc; /* Calculation of the amplitude for n = 0, i.e. 2^0 = 1 time step. */
   err[0] = 0; /* For n = 0, the MC error is zero, as there are no integrals to calculate. */

   fprintf(stdout, "%d %1.15le %1.15le\n", two[0], amp[0], err[0]); /* Printing the n = 0 output. */
   fflush(stdout);

   for(n = 1; n <= s; n++) /* Calculation and printing of the amplitudes and errors for n > 0. */
   {
      amp[n] /= Nmc;
      err[n] = (err[n] / Nmc - amp[n] * amp[n]) / (Nmc - 1);

      if(err[n] > 0) err[n] = sqrt(err[n]); /* Avoiding roundoff error. */
      else err[n] = 0;

      fprintf(stdout, "%d %1.15le %1.15le\n", two[n], amp[n], err[n]);
      fflush(stdout);
   }

   /* Printing tcpu and Nmc. */
   fprintf(stdout, "Overall execution time: %le seconds.\n", tcpu);
   fprintf(stdout, "Number of Monte Carlo samples: %le.\n", Nmc);
   fflush(stdout);

   /* Deallocation of memory. */
   free_sprng(stream);
   free_dvector(dsigma2inv, 0, s);
   free_dvector(dsigma2, 0, s);
   free_dvector(q, 0, two[s]);
   free_dvector(distrexp, 0, two[s]);
   free_dvector(norm, 0, s);
   free_ivector(two, 0, s);
   free_dvector(err, 0, s);
   free_dvector(amp, 0, s);
   free_dvector(par, 0, argc - 8);
   free_dvector(deltas, 0, 2 * MAXDER);
   free_dvector(epss, 0, MAXDER);
   free_dvector(ftinv, 0, MAXDER);  
   free_dvector(ts, 0, MAXDER); 
   free_dvector(xs, 0, MAXDER);

   exit(EXIT_SUCCESS);
}

/* The function mc() does the main Monte Carlo loop. In the loop it calls the
   function distr() to generate the path, and enters the loop over the number
   of time steps. In this loop it calls the function func() to calculate
   f / p (for a given n), the quantity we are averaging to obtain the integral
   of the function f. The array *amp accumulates f / p values for each n, while the
   array *err accumulates (f / p)^2 values, needed in order to calculate the error.
*/

void mc(double Nmc, double *q, double *amp, double *err, int s)
{
   double i; /* Loop variable for MC samples. It is of the double type to allow for
                large Nmc values. */
   int n; /* Loop variable for the number of time steps. */
   double foverp; /* f / p value, returned by the func(). */

   for(i = 0; i < Nmc; i++) /* Loop over the MC samples. */
   {
      distr(q, s); /* Generating the path, with 2^s time steps. */

      for(n = 0; n <= s; n++) /* Loop over the number of time steps, 2^0, ..., 2^s. */
      {
         foverp = func(q, s, n); /* Calculates f / p for a given n. */
         amp[n] += foverp; /* Accumulates f / p for a given n. */
         err[n] += foverp * foverp; /* Accumulates (f / p)^2 for a given n. */
      }
   }

   return;
}

/* Function distr() generates the paths as arrays (q[0], ..., q[2^s]) using
   the bisection algorithm.
*/

void distr(double *q, int s)
{
   int n; /* Loop variable for bisection level. */
   int i; /* Loop variable for path nodes that need to be generated at bisection level n. */
   double temp; /* Temporary variable. */

   for(n = 1; n <= s; n++) /* Loop over bisection level. */
   {
      for(i = 0; i < two[n - 1]; i++) /* Loop over path nodes that need to be generated at */
      {                                /* bisection level n. */

         /* Using the Box-Mueller method to generate the random number temp from the Gaussian centered
            at zero, with 2 sigma^2 = dsigma2[n]. */
         temp = sqrt( - dsigma2[n] * log(sprng(stream))) * cos(dpi * sprng(stream));

         /* At bisection level n, the path node is centered at the mid-point of the nearest path nodes
            generated at bisection level n - 1. */
         q[(1 + 2 * i) * two[s - n]] = 0.5 * (q[i * two[s - n + 1]] + q[(1 + i) * two[s - n + 1]]) + temp;

         /* The exponent of the Gaussian probability density function. It will be substracted from the action in
            func(), which should return f / p. The normalization of the Gaussian is taken care of in *norm. */
         distrexp[(1 + 2 * i) * two[s - n]]  = temp * temp * dsigma2inv[n];
      }
   }

   return;
}
